<?php
/* --------------------------------------------------------------
   TrackingCodeApiRequestParser.php 2020-03-10
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2020 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

declare(strict_types=1);

namespace Gambio\Api\TrackingCode;

use DateTime;
use Exception;
use Gambio\Admin\TrackingCode\SqlCriteria\TrackingCodeFilter;
use Gambio\Admin\TrackingCode\SqlCriteria\TrackingCodeFilters;
use Gambio\Admin\TrackingCode\SqlCriteria\TrackingCodeSorting;
use Gambio\Admin\TrackingCode\SqlCriteria\TrackingCodeSortings;
use Gambio\Admin\TrackingCode\SqlCriteria\TrackingCodeSqlCriteria;
use Gambio\Admin\TrackingCode\SqlCriteria\TrackingCodeSqlPagination;
use Psr\Http\Message\ServerRequestInterface;

/**
 * Class TrackingCodeApiRequestParser
 *
 * @package Gambio\Api\TrackingCode
 */
class TrackingCodeApiRequestParser
{
    /**
     * @param ServerRequestInterface $request
     *
     * @return int
     */
    public function getPerPageFromRequest(ServerRequestInterface $request): int
    {
        return (int)$request->getQueryParam('per-page', 25);
    }
    
    
    /**
     * @param ServerRequestInterface $request
     *
     * @return int
     */
    public function getPageFromRequest(ServerRequestInterface $request): int
    {
        return (int)$request->getQueryParam('page', 1);
    }
    
    
    /**
     * @param ServerRequestInterface $request
     *
     * @return array
     */
    public function getFieldsFromRequest(ServerRequestInterface $request): array
    {
        $fields = $request->getQueryParam('fields');
        
        return ($fields === null) ? [] : explode(',', $fields);
    }
    
    
    /**
     * @param ServerRequestInterface $request
     *
     * @return TrackingCodeSqlPagination
     */
    public function getPaginationFromRequest(ServerRequestInterface $request): TrackingCodeSqlPagination
    {
        $perPage = $this->getPerPageFromRequest($request);
        $page    = $this->getPageFromRequest($request);
        
        return TrackingCodeSqlPagination::create($perPage, ($page - 1) * $perPage);
    }
    
    
    /**
     * @param ServerRequestInterface $request
     *
     * @return TrackingCodeSqlCriteria
     */
    public function getCriteriaFromRequest(ServerRequestInterface $request): TrackingCodeSqlCriteria
    {
        try {
            $createdAfter = $request->getQueryParam('created-after');
            $createdAfter = ($createdAfter !== null) ? new DateTime($createdAfter) : $createdAfter;
        } catch (Exception $exception) {
            $createdAfter = null;
        }
        
        return TrackingCodeSqlCriteria::create($this->getFiltersFromRequest($request),
                                               $this->getSortingFromRequest($request),
                                               $createdAfter);
    }
    
    
    /**
     * @param ServerRequestInterface $request
     *
     * @return TrackingCodeFilters
     */
    private function getFiltersFromRequest(ServerRequestInterface $request): TrackingCodeFilters
    {
        $filters = [];
        foreach ($request->getQueryParam('filter', []) as $attribute => $value) {
            $operation = (strpos($value, '*') !== false) ? 'like' : 'eq';
            if (strpos($value, '|') > 0) {
                $operation = substr($value, 0, strpos($value, '|'));
                $value     = substr($value, strpos($value, '|') + 1);
            }
            
            $filters[] = TrackingCodeFilter::create($attribute, $value, $operation);
        }
        
        return TrackingCodeFilters::create(...$filters);
    }
    
    
    /**
     * @param ServerRequestInterface $request
     *
     * @return TrackingCodeSortings
     */
    private function getSortingFromRequest(ServerRequestInterface $request): TrackingCodeSortings
    {
        $sortings = [];
        if ($request->getQueryParam('sort') !== null) {
            foreach (explode(',', $request->getQueryParam('sort')) as $attribute) {
                $order      = (strpos($attribute, '-') === 0) ? 'desc' : 'asc';
                $sortings[] = TrackingCodeSorting::create(ltrim($attribute, '+-'), $order);
            }
        }
        
        return TrackingCodeSortings::create(...$sortings);
    }
    
    
    /**
     * @param ServerRequestInterface $request
     *
     * @return string
     */
    public function getResourceUrlFromRequest(ServerRequestInterface $request): string
    {
        return $request->getUri()->getScheme() . '://' . $request->getUri()->getHost() . $request->getUri()->getPath();
    }
}